import { Link, useLocation } from "wouter";
import { useI18n } from "@/lib/i18n";
import { Menu, X, Globe, ShoppingCart, Trash2, Plus, Minus } from "lucide-react";
import { useState } from "react";
import { useCart } from "@/lib/cart";
import { Sheet, SheetContent, SheetHeader, SheetTitle, SheetTrigger, SheetFooter } from "@/components/ui/sheet";
import { Button } from "@/components/ui/button";
import { Separator } from "@/components/ui/separator";
import logoImg from "@assets/WhatsApp_Image_2025-12-21_at_1.54.00_PM_1766309101495.jpeg";

export function Navbar() {
  const { t, language, setLanguage } = useI18n();
  const { items, updateQuantity, removeFromCart, totalPrice, totalItems } = useCart();
  const [location] = useLocation();
  const [isOpen, setIsOpen] = useState(false);

  const toggleLang = () => {
    setLanguage(language === 'en' ? 'mr' : 'en');
  };

  const navLinks = [
    { href: "/", label: "nav.home" },
    { href: "/menu", label: "nav.menu" },
    { href: "/subscriptions", label: "nav.subscriptions" },
    { href: "/franchises", label: "nav.franchises" },
    { href: "/contact", label: "nav.contact" },
  ];

  return (
    <nav className="sticky top-0 z-50 w-full bg-white/90 backdrop-blur-md border-b border-border shadow-sm">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex justify-between items-center h-20">
          
          {/* Logo */}
          <Link href="/" className="flex-shrink-0 flex items-center gap-3 group">
            <div className="relative overflow-hidden rounded-full border-2 border-primary/20 group-hover:border-primary transition-colors duration-300">
              <img src={logoImg} alt="Daily Bowl Logo" className="h-12 w-12 object-cover" />
            </div>
            <span className="font-display font-bold text-2xl text-primary tracking-tight">Daily Bowl</span>
          </Link>

          {/* Desktop Nav */}
          <div className="hidden md:flex items-center space-x-8">
            {navLinks.map((link) => (
              <Link 
                key={link.href} 
                href={link.href} 
                className={`font-medium text-base transition-colors duration-200 ${
                  location === link.href 
                    ? "text-primary font-semibold" 
                    : "text-foreground/70 hover:text-accent"
                }`}
              >
                {t(link.label)}
              </Link>
            ))}

            <button
              onClick={toggleLang}
              className="flex items-center gap-2 px-4 py-2 rounded-full bg-secondary/50 hover:bg-secondary text-secondary-foreground font-medium transition-all"
            >
              <Globe className="w-4 h-4" />
              <span>{language === 'en' ? 'मराठी' : 'English'}</span>
            </button>

            {/* Mobile Cart Trigger */}
            <div className="md:hidden">
              <Sheet>
                <SheetTrigger asChild>
                  <button className="relative p-2 text-foreground/70 hover:text-primary transition-colors">
                    <ShoppingCart className="w-6 h-6" />
                    {totalItems > 0 && (
                      <span className="absolute -top-1 -right-1 bg-accent text-white text-[10px] font-bold w-5 h-5 flex items-center justify-center rounded-full shadow-sm">
                        {totalItems}
                      </span>
                    )}
                  </button>
                </SheetTrigger>
                <SheetContent className="flex flex-col w-full sm:max-w-md p-0 rounded-l-[2rem]">
                  <SheetHeader className="p-6 border-b">
                    <SheetTitle className="text-2xl font-display font-bold text-primary flex items-center gap-2">
                      <ShoppingCart className="w-6 h-6" />
                      Your Bowl Cart
                    </SheetTitle>
                  </SheetHeader>
                  
                  <div className="flex-grow overflow-y-auto p-6 space-y-6">
                    {items.length === 0 ? (
                      <div className="flex flex-col items-center justify-center h-full text-center text-gray-400">
                        <ShoppingCart className="w-16 h-16 mb-4 opacity-10" />
                        <p className="text-lg font-medium">Your cart is empty</p>
                        <Link href="/menu" className="text-primary hover:underline mt-2">Browse the Menu</Link>
                      </div>
                    ) : (
                      items.map((item) => (
                        <div key={item.id} className="flex gap-4">
                          <div className="w-20 h-20 rounded-2xl overflow-hidden bg-gray-100 shrink-0">
                            <img src={item.imageUrl} alt={item.name} className="w-full h-full object-cover" />
                          </div>
                          <div className="flex-grow">
                            <div className="flex justify-between mb-1">
                              <h4 className="font-bold text-gray-900 line-clamp-1">
                                {language === 'mr' ? item.marathiName || item.name : item.name}
                              </h4>
                              <button onClick={() => removeFromCart(item.id)} className="text-gray-400 hover:text-red-500">
                                <Trash2 className="w-4 h-4" />
                              </button>
                            </div>
                            <div className="flex items-center justify-between">
                              <div className="flex items-center gap-3 bg-gray-100 rounded-full px-2 py-1">
                                <button onClick={() => updateQuantity(item.id, -1)} className="p-1 hover:text-primary"><Minus className="w-3 h-3" /></button>
                                <span className="text-sm font-bold w-4 text-center">{item.quantity}</span>
                                <button onClick={() => updateQuantity(item.id, 1)} className="p-1 hover:text-primary"><Plus className="w-3 h-3" /></button>
                              </div>
                              <span className="font-bold text-accent">₹{item.price * item.quantity}</span>
                            </div>
                          </div>
                        </div>
                      ))
                    )}
                  </div>

                  {items.length > 0 && (
                    <SheetFooter className="p-6 bg-gray-50 border-t flex-col gap-4">
                      <div className="flex justify-between items-center w-full">
                        <span className="text-gray-500 font-medium">Total Amount</span>
                        <span className="text-2xl font-display font-bold text-primary">₹{totalPrice}</span>
                      </div>
                      <Button className="w-full h-14 rounded-xl text-lg font-bold shadow-lg shadow-primary/25">
                        Checkout Now
                      </Button>
                    </SheetFooter>
                  )}
                </SheetContent>
              </Sheet>
            </div>

            {/* Desktop Cart Trigger */}
            <div className="hidden md:block">
              <Sheet>
                <SheetTrigger asChild>
                  <button className="relative p-2 text-foreground/70 hover:text-primary transition-colors">
                    <ShoppingCart className="w-6 h-6" />
                    {totalItems > 0 && (
                      <span className="absolute -top-1 -right-1 bg-accent text-white text-[10px] font-bold w-5 h-5 flex items-center justify-center rounded-full shadow-sm">
                        {totalItems}
                      </span>
                    )}
                  </button>
                </SheetTrigger>
                <SheetContent className="flex flex-col w-full sm:max-w-md p-0 rounded-l-[2rem]">
                  <SheetHeader className="p-6 border-b">
                    <SheetTitle className="text-2xl font-display font-bold text-primary flex items-center gap-2">
                      <ShoppingCart className="w-6 h-6" />
                      Your Bowl Cart
                    </SheetTitle>
                  </SheetHeader>
                  
                  <div className="flex-grow overflow-y-auto p-6 space-y-6">
                    {items.length === 0 ? (
                      <div className="flex flex-col items-center justify-center h-full text-center text-gray-400">
                        <ShoppingCart className="w-16 h-16 mb-4 opacity-10" />
                        <p className="text-lg font-medium">Your cart is empty</p>
                        <Link href="/menu" className="text-primary hover:underline mt-2">Browse the Menu</Link>
                      </div>
                    ) : (
                      items.map((item) => (
                        <div key={item.id} className="flex gap-4">
                          <div className="w-20 h-20 rounded-2xl overflow-hidden bg-gray-100 shrink-0">
                            <img src={item.imageUrl} alt={item.name} className="w-full h-full object-cover" />
                          </div>
                          <div className="flex-grow">
                            <div className="flex justify-between mb-1">
                              <h4 className="font-bold text-gray-900 line-clamp-1">
                                {language === 'mr' ? item.marathiName || item.name : item.name}
                              </h4>
                              <button onClick={() => removeFromCart(item.id)} className="text-gray-400 hover:text-red-500">
                                <Trash2 className="w-4 h-4" />
                              </button>
                            </div>
                            <div className="flex items-center justify-between">
                              <div className="flex items-center gap-3 bg-gray-100 rounded-full px-2 py-1">
                                <button onClick={() => updateQuantity(item.id, -1)} className="p-1 hover:text-primary"><Minus className="w-3 h-3" /></button>
                                <span className="text-sm font-bold w-4 text-center">{item.quantity}</span>
                                <button onClick={() => updateQuantity(item.id, 1)} className="p-1 hover:text-primary"><Plus className="w-3 h-3" /></button>
                              </div>
                              <span className="font-bold text-accent">₹{item.price * item.quantity}</span>
                            </div>
                          </div>
                        </div>
                      ))
                    )}
                  </div>

                  {items.length > 0 && (
                    <SheetFooter className="p-6 bg-gray-50 border-t flex-col gap-4">
                      <div className="flex justify-between items-center w-full">
                        <span className="text-gray-500 font-medium">Total Amount</span>
                        <span className="text-2xl font-display font-bold text-primary">₹{totalPrice}</span>
                      </div>
                      <Button className="w-full h-14 rounded-xl text-lg font-bold shadow-lg shadow-primary/25">
                        Checkout Now
                      </Button>
                    </SheetFooter>
                  )}
                </SheetContent>
              </Sheet>
            </div>
          </div>

          {/* Mobile Menu Button */}
          <div className="md:hidden flex items-center">
            <button
              onClick={() => setIsOpen(!isOpen)}
              className="text-foreground hover:text-primary transition-colors p-2"
            >
              {isOpen ? <X className="w-6 h-6" /> : <Menu className="w-6 h-6" />}
            </button>
          </div>
        </div>
      </div>

      {/* Mobile Menu */}
      {isOpen && (
        <div className="md:hidden bg-white border-t border-border">
          <div className="px-4 pt-2 pb-6 space-y-2">
            {navLinks.map((link) => (
              <Link 
                key={link.href} 
                href={link.href}
                onClick={() => setIsOpen(false)}
                className={`block px-3 py-3 rounded-lg text-lg font-medium ${
                  location === link.href 
                    ? "bg-primary/10 text-primary" 
                    : "text-foreground hover:bg-secondary/30 hover:text-accent"
                }`}
              >
                {t(link.label)}
              </Link>
            ))}
            <button
              onClick={() => {
                toggleLang();
                setIsOpen(false);
              }}
              className="w-full text-left px-3 py-3 mt-4 rounded-lg text-lg font-medium text-secondary-foreground bg-secondary/20 flex items-center gap-2"
            >
              <Globe className="w-5 h-5" />
              Change Language ({language === 'en' ? 'Marathi' : 'English'})
            </button>
          </div>
        </div>
      )}
    </nav>
  );
}
