import { useSubmitContact } from "@/hooks/use-contact";
import { useI18n } from "@/lib/i18n";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { insertContactRequestSchema, type InsertContactRequest } from "@shared/schema";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Phone, Mail, MapPin, Send } from "lucide-react";
import { motion } from "framer-motion";

export default function Contact() {
  const { t } = useI18n();
  const mutation = useSubmitContact();
  
  const form = useForm<InsertContactRequest>({
    resolver: zodResolver(insertContactRequestSchema),
    defaultValues: {
      name: "",
      phone: "",
      message: ""
    }
  });

  const onSubmit = (data: InsertContactRequest) => {
    mutation.mutate(data, {
      onSuccess: () => form.reset()
    });
  };

  return (
    <div className="min-h-screen bg-gray-50 pt-16 pb-24">
      {/* Header */}
      <div className="bg-primary pb-32 pt-16 px-4">
        <div className="max-w-4xl mx-auto text-center text-white">
          <h1 className="text-4xl md:text-5xl font-display font-bold mb-4">{t("contact.title")}</h1>
          <p className="text-white/80 text-lg">We'd love to hear from you. Reach out for bulk orders or feedback.</p>
        </div>
      </div>

      <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 -mt-20">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
          
          {/* Contact Info Card */}
          <motion.div 
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ delay: 0.2 }}
            className="bg-primary-foreground rounded-3xl p-8 shadow-xl border border-gray-100 flex flex-col justify-between min-h-[400px]"
          >
            <div>
              <h2 className="text-2xl font-display font-bold text-primary mb-8">Contact Information</h2>
              <div className="space-y-8">
                <div className="flex items-start gap-5">
                  <div className="w-12 h-12 rounded-2xl bg-primary/10 flex items-center justify-center shrink-0 text-primary">
                    <Phone className="w-6 h-6" />
                  </div>
                  <div>
                    <h3 className="font-bold text-gray-900 mb-1">Phone</h3>
                    <p className="text-gray-500 font-medium">+91 9594620104</p>
                    <p className="text-xs text-gray-400 mt-1">Mon-Sat, 9am - 7pm</p>
                  </div>
                </div>

                <div className="flex items-start gap-5">
                  <div className="w-12 h-12 rounded-2xl bg-accent/10 flex items-center justify-center shrink-0 text-accent">
                    <Mail className="w-6 h-6" />
                  </div>
                  <div>
                    <h3 className="font-bold text-gray-900 mb-1">Email</h3>
                    <p className="text-gray-500 font-medium">dailybowls03@gmail.com</p>
                    <p className="text-xs text-gray-400 mt-1">For corporate orders & queries</p>
                  </div>
                </div>

                <div className="flex items-start gap-5">
                  <div className="w-12 h-12 rounded-2xl bg-blue-50 flex items-center justify-center shrink-0 text-blue-600">
                    <MapPin className="w-6 h-6" />
                  </div>
                  <div>
                    <h3 className="font-bold text-gray-900 mb-1">Kitchen Location</h3>
                    <p className="text-gray-500 font-medium">Gawand baug, kokanipada, Thane west 400 610</p>
                  </div>
                </div>
              </div>
            </div>

            <div className="mt-12 pt-8 border-t border-gray-100">
              <p className="text-sm text-gray-400">
                Are you an influencer or brand? <a href="#" className="text-primary font-bold hover:underline">Partner with us</a>
              </p>
            </div>
          </motion.div>

          {/* Form Card */}
          <motion.div 
            initial={{ opacity: 0, x: 20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ delay: 0.3 }}
            className="bg-white rounded-3xl p-8 shadow-xl border border-gray-100"
          >
             <h2 className="text-2xl font-display font-bold text-gray-900 mb-6">{t("contact.submit")}</h2>
             
             <Form {...form}>
               <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
                 <FormField
                   control={form.control}
                   name="name"
                   render={({ field }) => (
                     <FormItem>
                       <FormLabel className="text-gray-600 font-medium">{t("contact.name")}</FormLabel>
                       <FormControl>
                         <Input placeholder="John Doe" className="h-12 rounded-xl bg-gray-50 border-gray-200 focus:bg-white" {...field} />
                       </FormControl>
                       <FormMessage />
                     </FormItem>
                   )}
                 />

                 <FormField
                   control={form.control}
                   name="phone"
                   render={({ field }) => (
                     <FormItem>
                       <FormLabel className="text-gray-600 font-medium">{t("contact.phone")}</FormLabel>
                       <FormControl>
                         <Input placeholder="+91 00000 00000" className="h-12 rounded-xl bg-gray-50 border-gray-200 focus:bg-white" {...field} />
                       </FormControl>
                       <FormMessage />
                     </FormItem>
                   )}
                 />

                 <FormField
                   control={form.control}
                   name="message"
                   render={({ field }) => (
                     <FormItem>
                       <FormLabel className="text-gray-600 font-medium">{t("contact.message")}</FormLabel>
                       <FormControl>
                         <Textarea 
                           placeholder="I would like to order..." 
                           className="min-h-[140px] rounded-xl bg-gray-50 border-gray-200 focus:bg-white resize-none" 
                           {...field} 
                         />
                       </FormControl>
                       <FormMessage />
                     </FormItem>
                   )}
                 />

                 <button
                   type="submit"
                   disabled={mutation.isPending}
                   className="w-full h-14 rounded-xl bg-accent hover:bg-accent/90 text-white font-bold text-lg shadow-lg shadow-accent/25 transition-all flex items-center justify-center gap-2 disabled:opacity-70 disabled:cursor-not-allowed"
                 >
                   {mutation.isPending ? (
                     t("contact.sending")
                   ) : (
                     <>
                       {t("contact.submit")}
                       <Send className="w-5 h-5" />
                     </>
                   )}
                 </button>
               </form>
             </Form>
          </motion.div>
        </div>
      </div>
    </div>
  );
}
