import { useI18n } from "@/lib/i18n";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Send, Briefcase, TrendingUp, Users, Award } from "lucide-react";
import { motion } from "framer-motion";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";

const franchiseApplicationSchema = z.object({
  businessOwnerName: z.string().min(3, "Business owner name is required"),
  businessName: z.string().min(3, "Business name is required"),
  email: z.string().email("Invalid email"),
  phone: z.string().min(10, "Valid phone number required"),
  city: z.string().min(2, "City is required"),
  experience: z.string().min(10, "Please describe your business experience"),
  investmentCapacity: z.enum(["low", "medium", "high"], {
    errorMap: () => ({ message: "Please select investment capacity" })
  }),
  message: z.string().min(20, "Please provide more details about your interest"),
});

type FranchiseApplicationForm = z.infer<typeof franchiseApplicationSchema>;

export default function Franchises() {
  const { t, language } = useI18n();
  const { toast } = useToast();

  const form = useForm<FranchiseApplicationForm>({
    resolver: zodResolver(franchiseApplicationSchema),
    defaultValues: {
      businessOwnerName: "",
      businessName: "",
      email: "",
      phone: "",
      city: "",
      experience: "",
      investmentCapacity: "medium",
      message: "",
    }
  });

  const onSubmit = async (data: FranchiseApplicationForm) => {
    try {
      const response = await apiRequest('POST', '/api/contact', {
        name: `${data.businessOwnerName} - ${data.businessName}`,
        phone: data.phone,
        message: `Franchise Inquiry\n\nBusiness Owner: ${data.businessOwnerName}\nBusiness Name: ${data.businessName}\nEmail: ${data.email}\nCity: ${data.city}\nExperience: ${data.experience}\nInvestment Capacity: ${data.investmentCapacity}\nAdditional Message: ${data.message}`,
      });

      if (response.ok) {
        toast({
          title: language === 'en' ? "Application Submitted!" : "अर्ज सबमिट केला!",
          description: language === 'en' 
            ? "We'll review your application and contact you soon." 
            : "आम्ही आपल्या अर्जाचे मूल्यांकन करू आणि लवकरच संपर्क साधू.",
        });
        form.reset();
      }
    } catch (error) {
      toast({
        title: language === 'en' ? "Error" : "त्रुटी",
        description: language === 'en' 
          ? "Failed to submit application. Please try again." 
          : "अर्ज सबमिट करण्यात अपयश. कृपया पुन्हा प्रयत्न करा.",
        variant: "destructive",
      });
    }
  };

  const benefits = [
    {
      icon: Briefcase,
      title: language === 'en' ? "Established Brand" : "स्थापित ब्रँड",
      desc: language === 'en' 
        ? "Join a trusted brand with proven success and recognition in the healthy food space." 
        : "आरोग्यकर भोजनात सिद्ध यशस्वी विश्वसनीय ब्रँडसोबत सामील व्हा.",
    },
    {
      icon: TrendingUp,
      title: language === 'en' ? "Growing Market" : "वाढते बाजार",
      desc: language === 'en' 
        ? "Tap into the rapidly growing health and wellness industry with strong demand." 
        : "मजबूत मागणीसह तेजीने वाढणाऱ्या आरोग्य आणि कल्याण उद्योगात प्रवेश करा.",
    },
    {
      icon: Users,
      title: language === 'en' ? "Full Support" : "पूर्ण समर्थन",
      desc: language === 'en' 
        ? "Get comprehensive training, marketing support, and operational guidance." 
        : "व्यापक प्रशिक्षण, विपणन समर्थन आणि परिचालन मार्गदर्शन मिळा.",
    },
    {
      icon: Award,
      title: language === 'en' ? "Quality Standards" : "गुणवत्ता मानक",
      desc: language === 'en' 
        ? "Maintain consistent quality and nutrition standards across all locations." 
        : "सर्व स्थानांवर सातत्यपूर्ण गुणवत्ता आणि पोषण मानक राखा.",
    },
  ];

  return (
    <div className="bg-gray-50/50 min-h-screen pb-20">
      {/* Header */}
      <div className="bg-primary pt-20 pb-24 px-4 text-center text-white relative overflow-hidden">
        <div className="absolute inset-0 bg-[url('https://images.unsplash.com/photo-1552664730-d307ca884978?q=80&w=2053&auto=format&fit=crop')] opacity-10 bg-cover bg-center" />
        <div className="relative z-10">
          <h1 className="text-4xl md:text-5xl font-display font-bold mb-4">
            {language === 'en' ? 'Daily Bowl Franchise' : 'डेली बाउल फ्रँचायझ'}
          </h1>
          <p className="text-white/80 max-w-xl mx-auto text-lg">
            {language === 'en'
              ? "Become a part of the healthy food revolution. Build your own business with Daily Bowl."
              : "आरोग्यकर खाद्य क्रांतीचा भाग बनण्यासाठी. डेली बाउलसह आपला स्वतःचा व्यवसाय तयार करा."}
          </p>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 -mt-10 relative z-20">
        
        {/* Benefits Section */}
        <div className="mb-20">
          <div className="text-center mb-12">
            <h2 className="text-3xl md:text-4xl font-display font-bold text-gray-900 mb-4">
              {language === 'en' ? 'Why Choose Daily Bowl Franchise?' : 'डेली बाउल फ्रँचायझ का निवडा?'}
            </h2>
          </div>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
            {benefits.map((benefit, idx) => {
              const Icon = benefit.icon;
              return (
                <motion.div
                  key={idx}
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: idx * 0.1 }}
                  className="bg-white rounded-3xl p-8 shadow-lg shadow-gray-200/50 border border-gray-100 hover:shadow-xl hover:border-primary/20 transition-all"
                >
                  <div className="w-14 h-14 bg-primary/10 rounded-2xl flex items-center justify-center mb-4">
                    <Icon className="w-7 h-7 text-primary" />
                  </div>
                  <h3 className="font-display font-bold text-lg text-gray-900 mb-2">{benefit.title}</h3>
                  <p className="text-gray-600 text-sm leading-relaxed">{benefit.desc}</p>
                </motion.div>
              );
            })}
          </div>
        </div>

        {/* Application Form Section */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-start">
          
          {/* Left side - Info */}
          <motion.div
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ delay: 0.2 }}
            className="bg-primary-foreground rounded-3xl p-8 shadow-xl border border-gray-100"
          >
            <h2 className="text-3xl font-display font-bold text-primary mb-8">
              {language === 'en' ? 'Ready to Apply?' : 'अर्ज करण्यासाठी तयार?'}
            </h2>
            
            <div className="space-y-8">
              <div className="space-y-3">
                <h3 className="font-bold text-lg text-gray-900">
                  {language === 'en' ? 'What We Look For' : 'आम्ही काय शोधतो'}
                </h3>
                <ul className="space-y-2">
                  <li className="flex items-center gap-3">
                    <div className="w-2 h-2 bg-primary rounded-full"></div>
                    <span className="text-gray-600">
                      {language === 'en' ? 'Entrepreneurial mindset' : 'उद्यमशील मानसिकता'}
                    </span>
                  </li>
                  <li className="flex items-center gap-3">
                    <div className="w-2 h-2 bg-primary rounded-full"></div>
                    <span className="text-gray-600">
                      {language === 'en' ? 'Strong business acumen' : 'मजबूत व्यावसायिक कौशल्य'}
                    </span>
                  </li>
                  <li className="flex items-center gap-3">
                    <div className="w-2 h-2 bg-primary rounded-full"></div>
                    <span className="text-gray-600">
                      {language === 'en' ? 'Commitment to quality' : 'गुणवत्तेसाठी प्रतिबद्धता'}
                    </span>
                  </li>
                  <li className="flex items-center gap-3">
                    <div className="w-2 h-2 bg-primary rounded-full"></div>
                    <span className="text-gray-600">
                      {language === 'en' ? 'Passion for healthy food' : 'आरोग्यकर भोजनाप्रति आवेग'}
                    </span>
                  </li>
                </ul>
              </div>

              <div className="space-y-3">
                <h3 className="font-bold text-lg text-gray-900">
                  {language === 'en' ? 'Quick Support Contacts' : 'त्वरित समर्थन संपर्क'}
                </h3>
                <div className="space-y-2">
                  <p className="text-gray-600">
                    <span className="font-bold">Email:</span> dailybowls03@gmail.com
                  </p>
                  <p className="text-gray-600">
                    <span className="font-bold">Phone:</span> +91 9594620104
                  </p>
                </div>
              </div>
            </div>
          </motion.div>

          {/* Right side - Form */}
          <motion.div
            initial={{ opacity: 0, x: 20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ delay: 0.3 }}
            className="bg-white rounded-3xl p-8 shadow-xl border border-gray-100"
          >
            <h2 className="text-2xl font-display font-bold text-gray-900 mb-8">
              {language === 'en' ? 'Application Form' : 'अर्जपत्र'}
            </h2>

            <Form {...form}>
              <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
                
                <FormField
                  control={form.control}
                  name="businessOwnerName"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel className="text-gray-600 font-medium">
                        {language === 'en' ? 'Your Name' : 'तुमचे नाव'}
                      </FormLabel>
                      <FormControl>
                        <Input 
                          placeholder={language === 'en' ? 'John Doe' : 'जॉन डो'} 
                          className="h-12 rounded-xl bg-gray-50 border-gray-200 focus:bg-white" 
                          {...field} 
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="businessName"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel className="text-gray-600 font-medium">
                        {language === 'en' ? 'Proposed Business Name' : 'प्रस्तावित व्यवसाय नाव'}
                      </FormLabel>
                      <FormControl>
                        <Input 
                          placeholder={language === 'en' ? 'Your Business Name' : 'आपल्या व्यवसायाचे नाव'} 
                          className="h-12 rounded-xl bg-gray-50 border-gray-200 focus:bg-white" 
                          {...field} 
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="email"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel className="text-gray-600 font-medium">
                        {language === 'en' ? 'Email' : 'ईमेल'}
                      </FormLabel>
                      <FormControl>
                        <Input 
                          type="email"
                          placeholder="you@example.com" 
                          className="h-12 rounded-xl bg-gray-50 border-gray-200 focus:bg-white" 
                          {...field} 
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="phone"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel className="text-gray-600 font-medium">
                        {language === 'en' ? 'Phone Number' : 'फोन नंबर'}
                      </FormLabel>
                      <FormControl>
                        <Input 
                          placeholder="+91 00000 00000" 
                          className="h-12 rounded-xl bg-gray-50 border-gray-200 focus:bg-white" 
                          {...field} 
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="city"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel className="text-gray-600 font-medium">
                        {language === 'en' ? 'City' : 'शहर'}
                      </FormLabel>
                      <FormControl>
                        <Input 
                          placeholder={language === 'en' ? 'Mumbai' : 'मुंबई'} 
                          className="h-12 rounded-xl bg-gray-50 border-gray-200 focus:bg-white" 
                          {...field} 
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="investmentCapacity"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel className="text-gray-600 font-medium">
                        {language === 'en' ? 'Investment Capacity' : 'गुंतवणूकीची क्षमता'}
                      </FormLabel>
                      <FormControl>
                        <select 
                          className="h-12 w-full rounded-xl bg-gray-50 border border-gray-200 px-4 focus:bg-white focus:outline-none focus:ring-2 focus:ring-primary/20"
                          {...field}
                        >
                          <option value="low">{language === 'en' ? 'Low (Below ₹10L)' : 'कमी (₹10L खाली)'}</option>
                          <option value="medium">{language === 'en' ? 'Medium (₹10L - ₹25L)' : 'मध्यम (₹10L - ₹25L)'}</option>
                          <option value="high">{language === 'en' ? 'High (Above ₹25L)' : 'जास्त (₹25L वर)'}</option>
                        </select>
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="experience"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel className="text-gray-600 font-medium">
                        {language === 'en' ? 'Business Experience' : 'व्यावसायिक अनुभव'}
                      </FormLabel>
                      <FormControl>
                        <Textarea 
                          placeholder={language === 'en' ? 'Tell us about your business experience...' : 'आपल्या व्यावसायिक अनुभवाबद्दल सांगा...'} 
                          className="min-h-[100px] rounded-xl bg-gray-50 border-gray-200 focus:bg-white resize-none" 
                          {...field} 
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="message"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel className="text-gray-600 font-medium">
                        {language === 'en' ? 'Why Daily Bowl?' : 'डेली बाउल का?'}
                      </FormLabel>
                      <FormControl>
                        <Textarea 
                          placeholder={language === 'en' ? 'Tell us why you want to partner with Daily Bowl...' : 'डेली बाउलसह भागीदारी का करू इच्छितात ते सांगा...'} 
                          className="min-h-[100px] rounded-xl bg-gray-50 border-gray-200 focus:bg-white resize-none" 
                          {...field} 
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <button
                  type="submit"
                  disabled={form.formState.isSubmitting}
                  className="w-full h-14 rounded-xl bg-accent hover:bg-accent/90 text-white font-bold text-lg shadow-lg shadow-accent/25 transition-all flex items-center justify-center gap-2 disabled:opacity-70 disabled:cursor-not-allowed"
                >
                  {form.formState.isSubmitting ? (
                    language === 'en' ? 'Submitting...' : 'सबमिट करत आहे...'
                  ) : (
                    <>
                      {language === 'en' ? 'Submit Application' : 'अर्ज सबमिट करा'}
                      <Send className="w-5 h-5" />
                    </>
                  )}
                </button>
              </form>
            </Form>
          </motion.div>
        </div>
      </div>
    </div>
  );
}
