import { db } from "./db";
import {
  salads,
  subscriptions,
  combos,
  contactRequests,
  type Salad,
  type InsertSalad,
  type Subscription,
  type InsertSubscription,
  type Combo,
  type InsertCombo,
  type ContactRequest,
  type InsertContactRequest,
  type Outlet,
  type InsertOutlet,
  outlets,
} from "@shared/schema";
import { eq } from "drizzle-orm";

export interface IStorage {
  getSalads(): Promise<Salad[]>;
  getSalad(id: number): Promise<Salad | undefined>;
  createSalad(salad: InsertSalad): Promise<Salad>;
  
  getSubscriptions(): Promise<Subscription[]>;
  getSubscription(id: number): Promise<Subscription | undefined>;
  createSubscription(subscription: InsertSubscription): Promise<Subscription>;
  
  getCombos(): Promise<Combo[]>;
  getCombo(id: number): Promise<Combo | undefined>;
  createCombo(combo: InsertCombo): Promise<Combo>;
  
  createContactRequest(request: InsertContactRequest): Promise<ContactRequest>;

  getOutlets(): Promise<Outlet[]>;
  createOutlet(outlet: InsertOutlet): Promise<Outlet>;
}

export class DatabaseStorage implements IStorage {
  async getSalads(): Promise<Salad[]> {
    return await db.select().from(salads);
  }

  async getSalad(id: number): Promise<Salad | undefined> {
    const [salad] = await db.select().from(salads).where(eq(salads.id, id));
    return salad;
  }

  async createSalad(salad: InsertSalad): Promise<Salad> {
    const [newSalad] = await db.insert(salads).values(salad).returning();
    return newSalad;
  }

  async getSubscriptions(): Promise<Subscription[]> {
    return await db.select().from(subscriptions);
  }

  async getSubscription(id: number): Promise<Subscription | undefined> {
    const [subscription] = await db.select().from(subscriptions).where(eq(subscriptions.id, id));
    return subscription;
  }

  async createSubscription(subscription: InsertSubscription): Promise<Subscription> {
    const [newSubscription] = await db.insert(subscriptions).values(subscription).returning();
    return newSubscription;
  }

  async getCombos(): Promise<Combo[]> {
    return await db.select().from(combos);
  }

  async getCombo(id: number): Promise<Combo | undefined> {
    const [combo] = await db.select().from(combos).where(eq(combos.id, id));
    return combo;
  }

  async createCombo(combo: InsertCombo): Promise<Combo> {
    const [newCombo] = await db.insert(combos).values(combo).returning();
    return newCombo;
  }

  async createContactRequest(request: InsertContactRequest): Promise<ContactRequest> {
    const [newRequest] = await db.insert(contactRequests).values(request).returning();
    return newRequest;
  }

  async getOutlets(): Promise<Outlet[]> {
    return await db.select().from(outlets);
  }

  async createOutlet(outlet: InsertOutlet): Promise<Outlet> {
    const [newOutlet] = await db.insert(outlets).values(outlet).returning();
    return newOutlet;
  }
}

export const storage = new DatabaseStorage();
